package net.returnvoid.graphics.grid;

import processing.core.PVector;

/*
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

/**
 * An interface for building two-dimensional grids. These grids should be
 * specified with a bounding box (x, y, width, height). Elements in the grid
 * should be accessible with an index and their should be method to get the
 * positions of all elements in the grid. Elements in the grid are abstracted by
 * a certain position. That position should be handled as a center location when
 * it comes to draw shapes on that position.
 * 
 * @author Diana Lange
 *
 */
public interface RGrid {

	/**
	 * Set to x-position of the grid.
	 * 
	 * @param x
	 *            The new x-position of the grid.
	 * @return The current grid object.
	 */
	public RGrid setX(float x);

	/**
	 * Set to y-position of the grid.
	 * 
	 * @param y
	 *            The new y-position of the grid.
	 * @return The current grid object.
	 */
	public RGrid setY(float y);

	/**
	 * Sets the padding (space between elements of the grid).
	 * 
	 * @param padding
	 *            The new padding of this grid.
	 * 
	 * @return The current grid object.
	 */
	public RGrid setPadding(float padding);

	/**
	 * Returns all positions of all elements in the grid.
	 * 
	 * @return The positions.
	 */
	public PVector[] getCoordinates();

	/**
	 * Returns the current x-position of the grid.
	 * 
	 * @return The x-position.
	 */
	public Float getX();

	/**
	 * Returns the current y-position of the grid.
	 * 
	 * @return The y-position.
	 */
	public Float getY();

	/**
	 * Returns the width of the element at the given index.
	 * 
	 * @return The width.
	 */
	public Float getWidth();

	/**
	 * Returns the current height of the grid.
	 * 
	 * @return The height.
	 */
	public Float getHeight();

	/**
	 * Returns the current width of the grid.
	 * 
	 * @param index
	 *            The index of the element.
	 * @return The width.
	 */
	public Float getWidth(Integer index);

	/**
	 * Returns the current height of the grid.
	 * 
	 * @param index
	 *            The index of the element.
	 * @return The height.
	 */
	public Float getHeight(Integer index);

	/**
	 * Returns the space between the elements of the grid.
	 * 
	 * @return The padding.
	 */
	public Float getPadding();

	/**
	 * Returns the number of elements in the grid. This is usefull if you want
	 * to iterate over all elements of the grid. Example:<br>
	 * <br>
	 * 
	 * <pre>
	 * for (int i = 0; i < myGrid.size(); i++) {
	 * 		float x = myGrid.getX(i);
	 * 		float y = myGrid.getY(i):
	 * }
	 * </pre>
	 * 
	 * @return The number of elements of the grid.
	 */
	public Integer size();

	/**
	 * Returns the x-position of the element at the given index. <b>index</b>
	 * should be in the range of this grid [0, myGrid.size()], otherwise the
	 * call of this function might produce strange results or will produce an
	 * ArrayIndexOutOfBoundsException. Usage:<br>
	 * <br>
	 * 
	 * <pre>
	 * for (int i = 0; i < myGrid.size(); i++) {
	 * 		float x = myGrid.getX(i);
	 * 		float y = myGrid.getY(i):
	 * }
	 * </pre>
	 * 
	 * @param index
	 *            The index of the element.
	 * @return The x-position of the element.
	 */
	public Float getX(Integer index);

	/**
	 * Returns the y-position of the element at the given index. <b>index</b>
	 * should be in the range of this grid [0, myGrid.size()], otherwise the
	 * call of this function might produce strange results or will produce an
	 * ArrayIndexOutOfBoundsException. Usage:<br>
	 * <br>
	 * 
	 * <pre>
	 * for (int i = 0; i < myGrid.size(); i++) {
	 * 		float x = myGrid.getX(i);
	 * 		float y = myGrid.getY(i):
	 * }
	 * </pre>
	 * 
	 * @param index
	 *            The index of the element.
	 * @return The y-position of the element.
	 */
	public Float getY(Integer index);
}
