package net.returnvoid.color;

/*
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

/**
 * A collection of valid color spaces.
 * 
 * @author Diana Lange
 *
 */
public enum ColorSpace {

	RGBColor, XYZColor, LabColor, HSBColor, LCHColor;

	/**
	 * Converts a given ColorSpace to its ColorDofferenceMeasure equivalent. The
	 * result will always be the Euclidean distance from the given ColorSpace.
	 * E.G. RGBColor will return RGBEuclidean, LabColor will return LabDeltaE
	 * and so on.
	 * 
	 * @return The ColorDofferenceMeasure of the given ColorSpace.
	 */
	public ColorDifferenceMeasure toColorDifferenceMeasure() {
		if (this == RGBColor) {
			return ColorDifferenceMeasure.RGBEuclidean;
		} else if (this == HSBColor) {
			return ColorDifferenceMeasure.HSBEuclidean;
		} else if (this == LabColor) {
			return ColorDifferenceMeasure.LabDeltaE;
		} else if (this == LCHColor) {
			return ColorDifferenceMeasure.LCHEuclidean;
		} else {
			return ColorDifferenceMeasure.XYZEuclidean;
		}
	}

	/**
	 * Checks if a ColorSpace is equal to a ColorDofferenceMeasure. They are
	 * equal if they are both members of the same color model. E.G. RGBColor
	 * will be equal to RGBMeanRed but not equal to LabDeltaE.
	 * 
	 * @param m
	 *            The ColorDifferenceMeasure to compare to.
	 * @return True if the ColorDifferenceMeasure is equal.
	 */
	public boolean equals(ColorDifferenceMeasure m) {

		ColorSpace other = m.toColorSpace();

		return this.name().equals(other.name());
	}
}
