package net.returnvoid.analytics;

import java.util.ArrayList;

/*
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

/**
 * An interface to build clusters.
 * 
 * @author Diana Lange
 */
public interface Cluster {

	/**
	 * Adds an element to the cluster.
	 * 
	 * @param newMember
	 *            The new cluster member.
	 * @return The current cluster object.
	 */
	public Cluster addElement(Object newMember);

	/**
	 * Removes an element from the cluster (if the given element is a member).
	 * 
	 * @param member
	 *            The element which should be removed.
	 * @return The current cluster object.
	 */
	public Cluster removeElement(Object member);

	/**
	 * The given element <b>member</b> switches to another cluster under certain
	 * circumstances. Returns if the given member actually changed its cluster.
	 * 
	 * @param member
	 *            The element which might switch the cluster.
	 * @param newCluster
	 *            The cluster where the element might switch to.
	 * @return True, if element has changed its cluster. Otherwise false.
	 */
	public boolean switchElement(Object member, Cluster newCluster);

	/**
	 * Updates the mean value based on the current cluster members.
	 * 
	 * @return The current cluster object.
	 */
	public Cluster updateMean();

	/**
	 * Caculates the loss / error of the cluster.
	 * 
	 * @return The error / loss value.
	 */
	public Float getLoss();

	/**
	 * Returns the current mean of the cluster.
	 * 
	 * @return The cluster's mean.
	 */
	public Object getMean();

	/**
	 * Returns all cluster members.
	 * 
	 * @return The cluster members.
	 */
	public ArrayList<?> getElements();

}
