
import net.returnvoid.graphics.shape.*;
import net.returnvoid.tools.*;


/**
 * This sketch demonstates how to create and draw shapes.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */


void setup() {
  size(1280, 720);
}


void draw() {

  background(240);
  stroke(30, 100, 130);
  strokeWeight(4);
  fill(20, 170, 200);

  // Line -------------------------------------------------------------

  // lines can be created by given start and end locations
  float lineX1 = 50;
  float lineY1 = 50;
  float lineX2 = 200;
  float lineY2 = 150;
  Line line1 = new Line(this, lineX1, lineY1, lineX2, lineY2);
  line1.draw();

  // or lines can be createt by a given start location, the angle and the length of the line
  PVector lineStart = new PVector(300, 50);
  float lineAngle = PI / 4;
  float lineLength = 200;
  Line line2 = new Line(this, lineStart, lineAngle, lineLength);
  line2.draw();

  // Rectangle -------------------------------------------------------------

  float rectX = 550;
  float rectY = 50;
  float rectWidth = 250;
  float rectHeight = 200;
  Rect rect = new Rect(this, rectX, rectY, rectWidth, rectHeight);
  rect.draw();

  // Ellipse -------------------------------------------------------------

  float ellipseCentereX = 1050;
  float ellipseCenterY = 150;
  float ellipseWidth = 300;
  float ellipseHeight = 200;
  Ellipse ellipse = new Ellipse(this, ellipseCentereX, ellipseCenterY, ellipseWidth, ellipseHeight);
  ellipse.draw();

  // LineSegments -------------------------------------------------------------

  // LineSegments is a continuous sequence of lines defined by an input set of coordinates
  noFill();
  PVector[] segCoords = {
    new PVector(100, 650), 
    new PVector(110, 600), 
    new PVector(140, 610), 
    new PVector(140, 500), 
    new PVector(190, 550), 
    new PVector(200, 450), 
    new PVector(250, 450), 
    new PVector(350, 525)
  };

  LineSegments seg = new LineSegments(this, segCoords);
  seg.draw();

  // Curve -------------------------------------------------------------

  // create a curve (Catmull-Rom Curve)that is defined by the input coordinates.
  // Curve uses Processing's implementation for curves and uses always Processing's
  // default values for curveTightness=0 and curveDetail=20.
  PVector[] curveCoords = {
    new PVector(100 + 400, 650), 
    new PVector(110 + 400, 600), 
    new PVector(140 + 400, 610), 
    new PVector(140 + 400, 500), 
    new PVector(190 + 400, 550), 
    new PVector(200 + 400, 450), 
    new PVector(250 + 400, 450), 
    new PVector(350 + 400, 525)
  };

  Curve curve = new Curve(this, curveCoords);
  curve.draw();

  // CurvedLens -------------------------------------------------------------

  //  create a CurvedLens. A CurvedLens is a lenses that is deformed by a curve
  // (Catmull-Rom Curves). The curve (defined by the input coordinates), that is used 
  // to build the CurveLens, will be smoothend so that the shape will have a resonable
  // and smooth outline.
  PVector[] lensCoords = {
    new PVector(100 + 800, 650), 
    new PVector(110 + 800, 600), 
    new PVector(140 + 800, 610), 
    new PVector(200 + 800, 450), 
    new PVector(250 + 800, 450), 
    new PVector(300 + 800, 500), 
    new PVector(350 + 800, 525)
  };

  float lensThickness = 100;
  CurvedLens lens = new CurvedLens(this, lensCoords, lensThickness);
  lens.draw();

  // get a LineSegments representation of the input coordinates (The LineSegments coordinates are equal to the input coordinates of the curved lens).
  strokeWeight(2);
  stroke(180);
  LineSegments inputCoords = lens.toLineSegments();
  inputCoords.draw();

  // get a Curve representation of the input coordinates. The resulting curve is a smoothend version of the input coordinates and the returned
  // curve is the same curve that is used to build the outline of the shape.
  stroke(30, 100, 130, 180);
  Curve centerCurve = lens.toSmoothedCurve();
  centerCurve.draw();

  // Coordinates of shapes -------------------------------------------------------------

  // the coordinates, that define the shape, can be accessed by iteration 

  strokeWeight(6);
  stroke(180);
  drawCoordinates(inputCoords);

  stroke(30, 100, 130, 120);
  drawCoordinates(centerCurve);

  // put all created shapes in an array
  RShape[] shapes = {
    line1, 
    line2, 
    rect, 
    ellipse, 
    seg, 
    curve, 
    lens
  };

  // iterate over alle shapes of the array and draw their coordinates
  strokeWeight(10);
  stroke(240, 40, 150, 120);
  for (RShape shape : shapes) {
    drawCoordinates(shape);
  }
}

void drawCoordinates(RShape shape) {
  
  // iterate over all coordinates of the shape
  for (int i = 0; i < shape.size(); i++) {
    
    // get the current coordinate
    PVector p = shape.get(i);
    point(p.x, p.y);
  }
}

void keyPressed() {
  saveFrame("export/" + StringTools.timestamp() + ".png");
}