
import net.returnvoid.graphics.shape.*;
import net.returnvoid.tools.*;


/**
 * This sketch demonstates how to create a CurvedLens shape, how to manipulate it and how to draw it.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

void setup() {
  size(1280, 720);
}


void draw() {

  background(240);

  // coordinates for curved lens
  PVector[] locations = {
    new PVector(256, 560), 
    new PVector(289, 541), 
    new PVector(327, 534), 
    new PVector(361, 553), 
    new PVector(400, 556), 
    new PVector(437, 551), 
    new PVector(476, 551), 
    new PVector(512, 562), 
    new PVector(547, 545), 
    new PVector(583, 560)
  };


  // build the lens and move it to target location (top left lens)
  CurvedLens lens = new CurvedLens(this, locations, 60);
  lens.setX(100);
  lens.setY(100);
  drawLens(lens);

  // build a lens with same input coordinates but with more control points (top center lens)
  // detailScale=2 means that for each input coordinate one addional controlpoint is added
  CurvedLens moreDetails = lens.copy();
  moreDetails.setDetailScale(2);
  moreDetails.translate(400, 0);
  drawLens(moreDetails);

  // build a lens with same input coordinates but with less control points (top right lens)
  // details=8 means that 8 control points will be used for the lens construction
  CurvedLens lessDetails = lens.copy();
  lessDetails.setDetails(8);
  lessDetails.translate(800, 0);
  drawLens(lessDetails);

  // build a lens with same input coordinates but with more volume (bottom left lens)
  CurvedLens thick = lens.copy();
  thick.setHeight(120);
  thick.translate(0, 400);
  drawLens(thick);

  // build a lens with same input coordinates but with less length (bottom center lens)
  CurvedLens shortend = lens.copy();
  shortend.setLength(lens.getLength() * 0.5);
  shortend.translate(400, 400);
  drawLens(shortend);

  // build a lens with same input coordinates but rotated (bottom right lens)
  CurvedLens rotated = lens.copy();
  rotated.setRotation(-PI / 4);
  rotated.translate(800, 500);
  drawLens(rotated);
}

void drawLens(CurvedLens lens) {

  // draw lens
  fill(255);
  stroke(0);
  lens.draw();

  // draw the center curve (the center curve is used to construct the outline of the shape)
  Curve centerOfLens = lens.toSmoothedCurve();
  noFill();
  centerOfLens.draw();

  // iterate over all points of centerOfLens
  for (int i = 0; i < centerOfLens.size(); i++) {
    PVector p = centerOfLens.get(i);
    ellipse(p.x, p.y, 10, 10);
  }
}

void keyPressed() {
  saveFrame("export/" + StringTools.timestamp() + ".png");
}