import net.returnvoid.graphics.grid.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows how to create and use spiral grid systems.
 * The order of drawing will be effected by the set start and end radius for the spiral.
 * - if startRadius > endRadius: The elements in the center will be drawn last
 * - if startRadius < endRadius: The elements in the center will be drawn first
 * The rotations direction of the spiral can be changed: rotationDirection=1 equals to clockwise rotation, rotationDirection=-1  equals to anti-clickwise rotation
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

void setup() {
  size(1280, 720);
}

void draw() {

  background(#718F9A);

  strokeWeight(2);
  stroke(#225159);
  fill(#B9C0B8);

  // (1) Top left (simple) spiral: startRadius < endRadius and clockwise rotation ----------------------------------

  // set up the dimensions of the grid
  float gridX = width * 0.25;
  float gridY = height * 0.25;

  // start and end of spiral: elements in center will be drawn first
  float endRadius = height * 0.23;
  float startRadius = height * 0.05;

  // clockwise rotation
  int rotationDirection = 1;

  // position of first element
  float startAngle = map(mouseY, 0, height, 0, TWO_PI);

  // the spiral will rotate 4.75 times and the
  // elements will be separated by PI * 0.1 angle
  float gridAngle = PI * 0.1;
  float rotations = 4.75;

  // build and draw grid
  SpiralGrid spiralTopLeft = GridMaker.createSpiral(gridX, gridY, gridAngle, rotations);
  spiralTopLeft.setStartRadius(startRadius);
  spiralTopLeft.setEndRadius(endRadius);
  spiralTopLeft.setRotationDirection(rotationDirection);
  spiralTopLeft.setStartAngle(startAngle);
  drawGrid(spiralTopLeft);

  // (2) Top bottom spiral: startRadius < endRadius and clockwise rotation ----------------------------------

  // same settings as befor but with EquiSpiralGrid instead of SpiralGrid
  gridY = height * 0.75;

  // build and draw grid
  EquiSpiralGrid spiralBottomLeft = GridMaker.createEquiSpiral(gridX, gridY);
  spiralBottomLeft.setElementDistance(20);
  spiralBottomLeft.setRotations(rotations);
  spiralBottomLeft.setStartRadius(startRadius);
  spiralBottomLeft.setEndRadius(endRadius);
  spiralBottomLeft.setRotationDirection(rotationDirection);
  spiralBottomLeft.setStartAngle(startAngle);
  drawGrid(spiralBottomLeft);

  // (3) Top right (simple) spiral: startRadius > endRadius and anti clockwise rotation ----------------------------------

  // set up second grid which rotates in the other direction and the elemtents in
  // center will be drawn last
  gridY = height * 0.25;
  gridX = width * 0.75;

  // start and end of spiral: elements in center will be drawn last
  endRadius = height * 0.05;
  startRadius = height * 0.23;

  // anti clockwise rotation
  rotationDirection = -1;

  // build and draw grid
  SpiralGrid spiralTopRight = GridMaker.createSpiral(gridX, gridY, gridAngle, rotations);
  spiralTopRight.setStartRadius(startRadius);
  spiralTopRight.setEndRadius(endRadius);
  spiralTopRight.setRotationDirection(rotationDirection);
  spiralTopRight.setStartAngle(-startAngle);
  drawGrid(spiralTopRight);

  // (4) Top bottom spiral: startRadius > endRadius and anti clockwise rotation ----------------------------------

  // same settings as befor but with EquiSpiralGrid instead of SpiralGrid
  gridY = height * 0.75;

  // build and draw grid
  EquiSpiralGrid spiralBottomRight = GridMaker.createEquiSpiral(gridX, gridY);
  spiralBottomRight.setElementDistance(20);
  spiralBottomRight.setRotations(rotations);
  spiralBottomRight.setStartRadius(startRadius);
  spiralBottomRight.setEndRadius(endRadius);
  spiralBottomRight.setRotationDirection(rotationDirection);
  spiralBottomRight.setStartAngle(-startAngle);
  drawGrid(spiralBottomRight);
}

void drawGrid(RGrid grid) {

  // iterate of the elements of the grid
  for (int i = 0; i < grid.size(); i++) {

    float amt = map(i, 0, grid.size(), 0, 1);
    color fillColor = lerpColor(#B9C0B8, #225159, amt);
    color strokeColor = lerpColor(#225159, #B9C0B8, amt);

    fill(fillColor);
    stroke(strokeColor);

    // get location and dimension of the current element
    float x = grid.getX(i);
    float y = grid.getY(i);
    float w = grid.getWidth(i);
    float h = grid.getHeight(i);

    // draw ellipse with the given location and dimension
    ellipse(x, y, w, h);
  }
}

void keyPressed() {
  saveFrame("export/" + StringTools.timestamp() + ".png");
}