import net.*;
import net.returnvoid.*;
import net.returnvoid.analytics.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.*;
import net.returnvoid.graphics.grid.*;
import net.returnvoid.graphics.shape.*;
import net.returnvoid.io.*;
import net.returnvoid.test.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows how to...
 * ... create random colors within a specific color space
 * ... convert Processing color / color Strings to a color object
 * ... convert color spaces
 * ... store color objects from various color space in one arrey
 * ... convert all colors of an array using ColorHelper
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

void setup() {
  size(1280, 720);

  textFont(createFont("Arial", 14));
  noLoop();
}

void draw() {
  background(255);

  // (1) Left row : initalisation and random colors for each color space -----------------------------------------

  // some color
  color myRed = color(210, 40, 20);

  // another color, defined as String with hexadecimal notaion
  // Strings have to be converted to color first
  String myBlueString = "#1133AA";

  // String to color
  color myBlue = ColorHelper.getColor(myBlueString);

  // color to RGBColor
  RGBColor redRGB = RGBColor.toRGB(myRed);
  RGBColor blueRGB = RGBColor.toRGB(myBlue);

  // create random colors for each color space
  RGBColor rgb = RGBColor.getRandomRGB(); 
  HSBColor hsb = HSBColor.getRandomHSB();  
  XYZColor xyz = XYZColor.getRandomXYZ();
  LabColor lab = LabColor.getRandomLab();
  LCHColor lch = LCHColor.getRandomLCH();

  // all color spaces are RColor, which means they can
  // be put together in arrays
  RColor[] colors = {redRGB, blueRGB, rgb, hsb, xyz, lab, lch};

  // iterate over all colors
  float w = width / 3.0;
  float h = height / (float) colors.length;
  for (int i = 0; i < colors.length; i++) {
    float x = 0;
    float y = i * h;

    // we don't know anymore which color space
    // the color is, but all color classes share
    // some methods, e.g. the getColor() 
    RColor colorObj = colors[i];

    // get color of the object and general information as String
    color c = colorObj.getColor();
    String txt = colorObj.toString();

    // show color and information
    fill(c);
    stroke(c);
    rect(x, y, w, h);

    noStroke();
    fill(247);
    rect(x, y, textWidth(txt) + 20, 24);
    fill(0);
    text(txt, x + 10, y + 16);
  }

  // (2) Middle row : color conversion --------------------------------------------------------------------

  // convert all colors to LCHColor
  for (int i = 0; i < colors.length; i++) {

    // Options: toRGB(), toHSB(), toLab(), toXYZ(), toLCH()
    colors[i] = colors[i].toLCH();
  }

  // show converted colors again
  for (int i = 0; i < colors.length; i++) {
    float x = w;
    float y = i * h;

    RColor colorObj = colors[i];
    color c = colorObj.getColor();
    String txt = colorObj.toString();

    fill(c);
    stroke(c);
    rect(x, y, w, h);

    noStroke();
    fill(247);
    rect(x, y, textWidth(txt) + 20, 24);
    fill(0);
    text(txt, x + 10, y + 16);
  }

  // (3) Right row : color conversion with ColorHelper  -----------------------------------------------------

  // alternatively colors can be converted with ColorHelper.convert() and a given ColorSpace
  for (int i = 0; i < colors.length; i++) {

    // ColorSpaces are: RGBColor, HSBColor, LabColor, XYZColor, LCHColor
    colors[i] = ColorHelper.convert(colors[i], ColorSpace.RGBColor);
  }

  // or without loop in one line: ColorHelper.convert(colors, ColorSpace.RGBColor);

  // show converted colors again
  for (int i = 0; i < colors.length; i++) {
    float x = 2 * w;
    float y = i * h;

    RColor colorObj = colors[i];
    color c = colorObj.getColor();
    String txt = colorObj.toString();

    fill(c);
    stroke(c);
    rect(x, y, w, h);

    noStroke();
    fill(247);
    rect(x, y, textWidth(txt) + 20, 24);
    fill(0);
    text(txt, x + 10, y + 16);
  }
}


void mousePressed() {
  redraw();
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}