import net.returnvoid.color.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows the general workflow with
 * the color classes: 
 * - how to create instances for each color space, 
 * - how to access the color value for drawing with this color (e.g. use it with fill()), 
 * - how to change the values for the color channels and 
 * - how to get the values of the color channels.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

void setup() {
  size(1280, 720);

  textFont(createFont("Arial", 16));
}

void draw() {
  background(255);

  // (1) setup colors ----------------------------------------------

  // create one color for each color space 
  RGBColor rgb = new RGBColor(255, 120, 20);   // (red, green, blue)
  HSBColor hsb = new HSBColor(310, 88, 30);    // (hue, saturation, brightness)
  XYZColor xyz = new XYZColor(88, 55, 44);     // (x, y, z)
  LabColor lab = new LabColor(55, 6, 5);       // (luminance, a, b)
  LCHColor lch = new LCHColor(30, 88, 310);    // (luminance, chroma, hue)
  
  // get the color value for each object
  color rgbColor = rgb.getColor();
  color hsbColor = hsb.getColor();
  color xyzColor = xyz.getColor();
  color labColor = lab.getColor();
  color lchColor = lch.getColor();

  float y = 0;
  float w = width / 5.0;
  float h = height / 3.0;


  // draw all colors
  noStroke();
  fill(rgbColor); 
  rect(0, y, w, h);

  fill(hsbColor); 
  rect(w, y, w, h);

  fill(xyzColor); 
  rect(w * 2, y, w, h);

  fill(labColor); 
  rect(w * 3, y, w, h);

  fill(lchColor); 
  rect(w * 4, y, w, h);

  // the color space as well as the values of each color
  // channel and a hexadecimal representation can be 
  // accessed through following getters
  // getHexString() includes alpha channel, getOpHexString() excludes alpha channel ('Op' = Opaque) 
  String rgbText = rgb.getColorSpace() + 
    "\nred: " + rgb.getRed() + "\ngreen: " + rgb.getGreen() + "\nblue: " + rgb.getBlue() + "\nalpha: " + rgb.getAlpha() + 
    "\n" + rgb.getHexString() + " / " + rgb.getOpHexString();
  String hsbText = hsb.getColorSpace() + 
    "\nhue: " + hsb.getHue() + "\nsaturation: " + hsb.getSaturation() + "\nbrightness: " + hsb.getBrightness() + "\nalpha: " + hsb.getAlpha() +
    "\n" + hsb.getHexString() + " / " + hsb.getOpHexString();
  String xyzText = xyz.getColorSpace() + 
    "\nx: " + xyz.getX() + "\ny: " + xyz.getY() + "\nz: " + xyz.getZ() + "\nalpha: " + xyz.getAlpha() +
    "\n" + xyz.getHexString() + " / " + xyz.getOpHexString();
  String labText = lab.getColorSpace() + 
    "\nluminance: " + lab.getLuminance() + "\na: " + lab.getA() + "\nb: " + lab.getB() + "\nalpha: " + lab.getAlpha() +
    "\n" + lab.getHexString() + " / " + lab.getOpHexString();
  String lchText = lch.getColorSpace() + 
    "\nluminance: " + lch.getLuminance() + "\nchroma: " + lch.getChroma() + "\nhue: " + lch.getHue() + "\nalpha: " + lch.getAlpha() +
    "\n" + lch.getHexString() + " / " + lch.getOpHexString();

  // draw informations about each color
  fill(255);
  textAlign(CENTER, TOP);
  text(rgbText, w * 0.5, y + 10);
  text(hsbText, w * 1.5, y + 10);
  text(xyzText, w * 2.5, y + 10);
  text(labText, w * 3.5, y + 10);
  text(lchText, w * 4.5, y + 10);

  // (2) change colors ----------------------------------------------


  // the value for each color channel can be changed
  // with the setters (depending on color space)
  rgb.setRed(rgb.getRed() * 0.5);
  rgb.setBlue(rgb.getBlue() * 0.5);
  rgb.setGreen(rgb.getGreen() * 0.5);

  hsb.setHue(hsb.getHue() * 0.5);
  hsb.setSaturation(hsb.getSaturation() * 0.5);
  hsb.setBrightness(hsb.getBrightness() * 0.5);

  xyz.setX(xyz.getX() * 0.5);
  xyz.setY(xyz.getY() * 0.5);
  xyz.setZ(xyz.getZ() * 0.5);

  lab.setLuminance(lab.getLuminance() * 0.5);
  lab.setA(lab.getA() * 0.5);
  lab.setB(lab.getB() * 0.5); 

  lch.setLuminance(lch.getLuminance() * 0.5);
  lch.setChroma(lch.getChroma() * 0.5);
  lch.setHue(lch.getHue() * 0.5);

  y = h;

  // draw updated colors and their informations
  fill(rgb.getColor()); 
  rect(0, y, w, h);

  fill(hsb.getColor()); 
  rect(w, y, w, h);

  fill(xyz.getColor()); 
  rect(w * 2, y, w, h);

  fill(lab.getColor()); 
  rect(w * 3, y, w, h);

  fill(lch.getColor()); 
  rect(w * 4, y, w, h);

  rgbText = rgb.getColorSpace() + 
    "\nred: " + rgb.getRed() + "\ngreen: " + rgb.getGreen() + "\nblue: " + rgb.getBlue() + "\nalpha: " + rgb.getAlpha() + 
    "\n" + rgb.getHexString() + " / " + rgb.getOpHexString();
  hsbText = hsb.getColorSpace() + 
    "\nhue: " + hsb.getHue() + "\nsaturation: " + hsb.getSaturation() + "\nbrightness: " + hsb.getBrightness() + "\nalpha: " + hsb.getAlpha() +
    "\n" + hsb.getHexString() + " / " + hsb.getOpHexString();
  xyzText = xyz.getColorSpace() + 
    "\nx: " + xyz.getX() + "\ny: " + xyz.getY() + "\nz: " + xyz.getZ() + "\nalpha: " + xyz.getAlpha() +
    "\n" + xyz.getHexString() + " / " + xyz.getOpHexString();
  labText = lab.getColorSpace() + 
    "\nluminance: " + lab.getLuminance() + "\na: " + lab.getA() + "\nb: " + lab.getB() + "\nalpha: " + lab.getAlpha() +
    "\n" + lab.getHexString() + " / " + lab.getOpHexString();
  lchText = lch.getColorSpace() + 
    "\nluminance: " + lch.getLuminance() + "\nchroma: " + lch.getChroma() + "\nhue: " + lch.getHue() + "\nalpha: " + lch.getAlpha() +
    "\n" + lch.getHexString() + " / " + lch.getOpHexString();

  fill(255);
  textAlign(CENTER, TOP);
  text(rgbText, w * 0.5, y + 10);
  text(hsbText, w * 1.5, y + 10);
  text(xyzText, w * 2.5, y + 10);
  text(labText, w * 3.5, y + 10);
  text(lchText, w * 4.5, y + 10);

  // (3) change transparency ----------------------------------------------

  y = h * 2;

  // each color space has a getAlpha() and setAlpha(float) method
  // for RGB alpha should be in range of 0 - 255
  // for all other color spaces the range of alpha is 0 - 100
  rgb.setAlpha(127.5);
  hsb.setAlpha(50.0);
  xyz.setAlpha(50.0);
  lab.setAlpha(50.0);
  lch.setAlpha(50.0);

  // draw transparent colors and their values
  fill(rgb.getColor()); 
  rect(0, y, w, h);

  fill(hsb.getColor()); 
  rect(w, y, w, h);

  fill(xyz.getColor()); 
  rect(w * 2, y, w, h);

  fill(lab.getColor()); 
  rect(w * 3, y, w, h);

  fill(lch.getColor()); 
  rect(w * 4, y, w, h);

  rgbText = rgb.getColorSpace() + 
    "\nred: " + rgb.getRed() + "\ngreen: " + rgb.getGreen() + "\nblue: " + rgb.getBlue() + "\nalpha: " + rgb.getAlpha() + 
    "\n" + rgb.getHexString() + " / " + rgb.getOpHexString();
  hsbText = hsb.getColorSpace() + 
    "\nhue: " + hsb.getHue() + "\nsaturation: " + hsb.getSaturation() + "\nbrightness: " + hsb.getBrightness() + "\nalpha: " + hsb.getAlpha() +
    "\n" + hsb.getHexString() + " / " + hsb.getOpHexString();
  xyzText = xyz.getColorSpace() + 
    "\nx: " + xyz.getX() + "\ny: " + xyz.getY() + "\nz: " + xyz.getZ() + "\nalpha: " + xyz.getAlpha() +
    "\n" + xyz.getHexString() + " / " + xyz.getOpHexString();
  labText = lab.getColorSpace() + 
    "\nluminance: " + lab.getLuminance() + "\na: " + lab.getA() + "\nb: " + lab.getB() + "\nalpha: " + lab.getAlpha() +
    "\n" + lab.getHexString() + " / " + lab.getOpHexString();
  lchText = lch.getColorSpace() + 
    "\nluminance: " + lch.getLuminance() + "\nchroma: " + lch.getChroma() + "\nhue: " + lch.getHue() + "\nalpha: " + lch.getAlpha() +
    "\n" + lch.getHexString() + " / " + lch.getOpHexString();

  fill(0);
  textAlign(CENTER, TOP);
  text(rgbText, w * 0.5, y + 10);
  text(hsbText, w * 1.5, y + 10);
  text(xyzText, w * 2.5, y + 10);
  text(labText, w * 3.5, y + 10);
  text(lchText, w * 4.5, y + 10);
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}