import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;
import net.returnvoid.tools.*;

/**
 * This sketch demonstates how to calculate the mean color of a set of colors 
 * in any color space. Additionally the caluclation of loss is shown (distance / difference 
 * of the set of colors to the calculated mean). 
 *
 * Bonus: Find the color in the set that is the most similar to the mean.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

void setup() {
  size(1280, 720);

  textFont(createFont("Arial", 16));
  noLoop();
}

void draw() {
  background(255);

  strokeWeight(2);

  // set of colors - the mean color of these colors will be calculated
  RColor[] colors = {
    RGBColor.toRGB(#D6D8CD), 
    RGBColor.toRGB(#C0BAA2), 
    RGBColor.toRGB(#BFB4AE), 
    RGBColor.toRGB(#B59F8A), 
    RGBColor.toRGB(#C29B60), 
    RGBColor.toRGB(#BE7E46), 
    RGBColor.toRGB(#C17771), 
    RGBColor.toRGB(#A38877), 
    RGBColor.toRGB(#AD6A39), 
    RGBColor.toRGB(#6D776F), 
    RGBColor.toRGB(#D83A1A), 
    RGBColor.toRGB(#BF5221), 
    RGBColor.toRGB(#80655C), 
    RGBColor.toRGB(#9B5529), 
    RGBColor.toRGB(#635042), 
    RGBColor.toRGB(#635042), 
    RGBColor.toRGB(#4E2C19), 
    RGBColor.toRGB(#2E1D19)
  };

  // get a random color space (one of RGBColor, HSBColor, XYZColor, LCHColor, LabColor)
  ColorSpace space = ColorSpace.values()[RMath.random(ColorSpace.values().length)];

  // convert the color space to a difference measure (will always be
  // the euclidean distance of that color space, e.g. RGBEuclidean if space
  // is RGBColor
  ColorDifferenceMeasure measure = space.toColorDifferenceMeasure();

  // convert all colors of the set to the random choosen color space
  // it is possible to skip this step, but in most cases
  // the conversion makes the code run faster (no local conversion
  // of colors needed in the getMeanColor(), and getLoss() method).
  ColorHelper.convert(colors, space);


  textSize(14);
  textLeading(16);

  // draw circles for each color of the set 
  EllipseGrid grid = GridMaker.createEllipse(width * 0.5, height * 0.5, 300, colors.length);
  for (int i = 0; i < grid.size(); i++) {
    float x = grid.getX(i);
    float y = grid.getY(i);
    float d = grid.elementSize();

    // get the color channels
    // the channels are be channels[0]=red, channels[1]=green, channels[2]=blue, channels[3]=alpha for RGBColor
    // or [h, s, b, a] for HSBColor and so on 
    float[] channels = colors[i].getValues();
    String text = colors[i].getColorSpace() + "\n" + channels[0] + "\n" + channels[1] + "\n" + channels[2];

    // show color of the set and the values of the channels
    fill(colors[i].getColor());
    stroke(255);
    ellipse(x, y, d, d);

    fill(10, 20, 30);
    textAlign(CENTER, CENTER);
    text(text, x, y);
  }

  // calculate mean color in the given color space
  RColor mean = ColorHelper.getMeanColor(colors, space);
  
  // calculate the loss / error from the set of colors to the mean
  Float error = ColorHelper.getLoss(mean, colors, measure);

  // show mean color
  fill(mean.getColor());
  stroke(255);
  ellipse(width * 0.5, height * 0.5, 600 - grid.elementSize(), 600 - grid.elementSize());

  // show value of mean color and value of loss
  String text = "mean color:\n" + mean.toString() + "\n \nerror:\n" + error;

  textSize(16);
  fill(10, 20, 30);
  textAlign(CENTER, CENTER);
  text(text, width * 0.5, height * 0.5);
  
  // bonus: find the color in the set that is the most similar to the mean
  // the found color will have a gray outline, all other color have no outline
  int index = ColorDifference.getMostSimilarColor(mean, colors, measure);
  noFill();
  strokeWeight(4);
  stroke(200);
  ellipse(grid.getX(index), grid.getY(index), grid.elementSize(), grid.elementSize());
}


void mousePressed() {
  redraw();
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}