import net.returnvoid.analytics.*;
import net.returnvoid.color.*;
import net.returnvoid.io.*;
import net.returnvoid.tools.*;

/**
 * This sketchs demonstates how to create a quantization of an image (for each pixel the colors are replaced by a color of a palette).
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */


// the number of colors the palettes will have
int numberOfColors = 18;

// the measure defines how the colors are detected in the image
ColorDifferenceMeasure measure = ColorDifferenceMeasure.LabDeltaE;

// to container for the colors
ColorPalette palette;

// loader that handles the loading of images
ImageLoader loader;

// a container image (quantization will be applied to it)
PImage quantizedImg;

// a timestamp - time of quantization start. The calculation of the quantization can take a while
// if one is interested if the image is finished, this timestamp is needed (use palette.finishedImage(quantizedStamp) to see if image is fully quantized)
String quantizedStamp;

void setup() {
  size(1280, 720);
  
  textFont(createFont("Arial", 16));
  
  // create loader and get a image
  // a new image will be loaded on mouse click
  loader = new ImageLoader(this);
  loader.nextRandom();
  loader.disableKeys();
  
  // get 420 sampled colors from the image
  RColor[] training = ClusteringHelper.getTrainingColors(loader.getCurrent(), 420, measure);
  
  // get clustering (learn similarities in the sampled colors)
  ColorClustering clustering = ClusteringHelper.kmeans(training, numberOfColors, measure);
  
  // convert clustering to color palette
  palette = clustering.toColorPalette();
  
  // create a copy of the current input image and resize it
  quantizedImg = loader.getCurrent().copy();
  quantizedImg.resize(width / 2, 0);
  
  // start quantization of the copy image and save when the quantization started
  quantizedStamp = palette.getDithered(quantizedImg, 1);
}

void draw() {
  background(247);
  PImage org = loader.getCurrent();
  
  // draw image and color palette
  image(org, 0, 100, quantizedImg.width, quantizedImg.height);
  image(quantizedImg, width * 0.5, 100);
  
  // draw status of quantization
  textAlign(CENTER, TOP);
  String txt = "Done rendering quantization: " + palette.finishedImage(quantizedStamp);
  fill(10);
  text(txt, width * 0.5, quantizedImg.height + 110);
}

void mousePressed() {
  // get new image
  loader.nextRandom();
  
  // get 420 sampled colors from the image
  RColor[] training = ClusteringHelper.getTrainingColors(loader.getCurrent(), 420, measure);
  
  // get clustering (learn similarities in the sampled colors)
  ColorClustering clustering = ClusteringHelper.kmeans(training, numberOfColors, measure);
  
  // convert clustering to color palette
  palette = clustering.toColorPalette();
  
   // create a copy of the current input image and resize it
  quantizedImg = loader.getCurrent().copy();
  quantizedImg.resize(width / 2, 0);
  
  // start quantization of the copy image and save when the quantization started
  quantizedStamp = palette.get(quantizedImg);
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}