import net.returnvoid.color.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows how to merge / combine two Color Palettes.
 *
 * Controls:
 * Press mouse = raw merge and merge result with reduced number of colors
 * draw mouse = number of colors of merged palette
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

ColorPalette paletteOne;
ColorPalette paletteTwo;

void setup() {
  size(1280, 720);

  // load JSON data as Strings
  String[] paletteOneRawText = loadStrings("palettes/egon_schiele.color.json");
  String[] paletteTwoRawText = loadStrings("palettes/Vincent_Willem_van_Gogh_128_sunflowers.color.json");

  // join Strings and build color palettes from it
  paletteOne = new ColorPalette(join(paletteOneRawText, " "));
  paletteTwo = new ColorPalette(join(paletteTwoRawText, " "));

  // sort both palettes by hue
  paletteOne.sort("hue");
  paletteTwo.sort("hue");

  textFont(createFont("Arial", 16));
}

void draw() {

  background(10);
  textAlign(LEFT, TOP);

  float border = width * height * 0.00005;

  stroke(255);
  strokeWeight(4);

  // draw input color palettes
  rect(border, border, width * 0.5 - 1.5 * border, height * 0.5 - 1.5 * border);
  rect(width * 0.5 + 0.5 * border, border, width * 0.5 - 1.5 * border, height * 0.5 - 1.5 * border);
  paletteOne.draw(this, border, border, width * 0.5 - 1.5 * border, height * 0.5 - 1.5 * border);
  paletteTwo.draw(this, width * 0.5 + 0.5 * border, border, width * 0.5 - 1.5 * border, height * 0.5 - 1.5 * border);

  // merge palettes
  // because "paletteOne" should not be changed, it is copied and the copy gets merged with the second palette
  ColorPalette merged = paletteOne
    .copy()
    .merge(paletteTwo)
    .sort("hue");

  // draw merged palette
  rect(border, height * 0.5 + border * 0.5, width - 2 * border, height * 0.5 - 1.5 * border);

  if (!mousePressed) {
    
    // draw with all colors
    merged.draw(this, border, height * 0.5 + border * 0.5, width - 2 * border, height * 0.5 - 1.5 * border);
  } else {
    
    // draw with less colors
    
    // new number of colors for merged color palettes
    int newSize = (int) map(mouseX, 0, width, 3, merged.size());
    
    // reduce number of colors of pallete 
    // the color space of the palette is changed to improve similarity detection between colors
    merged
      .setColorSpace(ColorSpace.LabColor)
      .reduceSize(newSize)
      .sort("hue")
      .draw(this, border, height * 0.5 + border * 0.5, width - 2 * border, height * 0.5 - 1.5 * border);
  }
}

void mousePressed() {
  redraw();
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}