import net.returnvoid.color.*;
import net.returnvoid.tools.*;
import net.returnvoid.graphics.grid.*;

/**
 * This sketch shows how to get mixed colors from the palette. 
 * The mix is affected by the order of the colors as well as the color space of the palette.
 * 
 * Controls:
 * left mouse button : change color space of palette
 * right mouse button : change order of palette
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

// set up the colors of the palette
String paletteJSON = "{\"name\" : \"nemo\",\"tags\" : [],\"colors\" : [{\"84BBE4\" : {\"r\" : \"132.00424\",\"g\" : \"187.00317\",\"b\" : \"228.00114\",\"a\" : \"255.0\",\"importance\" : \"15\"}},{\"9FC7BF\" : {\"r\" : \"159.00197\",\"g\" : \"199.0041\",\"b\" : \"191.0015\",\"a\" : \"255.0\",\"importance\" : \"13\"}},{\"D78B98\" : {\"r\" : \"215.00021\",\"g\" : \"139.00562\",\"b\" : \"152.001\",\"a\" : \"255.0\",\"importance\" : \"46\"}},{\"A95E2D\" : {\"r\" : \"169.9995\",\"g\" : \"94.00521\",\"b\" : \"45.001495\",\"a\" : \"255.0\",\"importance\" : \"83\"}},{\"415C3D\" : {\"r\" : \"65.00039\",\"g\" : \"92.0021\",\"b\" : \"61.000984\",\"a\" : \"255.0\",\"importance\" : \"20\"}},{\"82495A\" : {\"r\" : \"130.00017\",\"g\" : \"73.00364\",\"b\" : \"90.000534\",\"a\" : \"255.0\",\"importance\" : \"18\"}},{\"6E3D1C\" : {\"r\" : \"110.999664\",\"g\" : \"61.003494\",\"b\" : \"28.000996\",\"a\" : \"255.0\",\"importance\" : \"72\"}},{\"354D73\" : {\"r\" : \"53.002853\",\"g\" : \"77.001236\",\"b\" : \"115.00049\",\"a\" : \"255.0\",\"importance\" : \"16\"}},{\"16151A\" : {\"r\" : \"22.000439\",\"g\" : \"21.0009\",\"b\" : \"26.000328\",\"a\" : \"255.0\",\"importance\" : \"137\"}}]}";
ColorPalette palette;

// all permitted color spaces
ColorSpace[] spaces = ColorSpace.values();

// index of current color space (in range if spaces[]);
int colorSpaceIndex = 0;

// all permitted orderings of a palette
String[] paletteSort = {"luminance", "hue", "huelch", "chroma", "red", "green", "blue", "saturation", "brightness"};

// index of current ordering (in range of paletteSort[])
int paletteSortIndex = 0;

void setup() {
  size(1280, 720);

  // create the palette object
  palette = new ColorPalette(paletteJSON);
  palette.resetImportance();

  textFont(createFont("Arial", 16));
}

void draw() {

  background(255);

  // draw raw palette
  palette.draw(this, 0, 0, width, height);

  // use a grid to draw small colored stripes to simulate a smooth gradient
  float padding = (float) width / palette.size();
  TableGrid grid = GridMaker.createTable(padding * 0.5, height * 0.5, width - padding, height * 0.5, 200, 1);

  float w = grid.cellWidth();
  float h = grid.cellHeight();

  // iterate over all cells of the grid
  noStroke();
  for (int i = 0; i < grid.size(); i++) {
    float x = grid.getX(i) - w * 0.5;
    float y = grid.getY(i) - h * 0.5;

    // calculate mixing parameter tplusi
    /*
     * tplusi:
     *            The parameter describes the color's position
     *            within the palette. It is calculated with t + i, where t is a
     *            float in range of [0, 1) and i is an int in range of [0,
     *            size() - 1). For example, if the color in the middle of the
     *            first and second color of the palette should be returned, than
     *            i=1 and t=0.5 and therefor tplusi=1.5.
     */
    float tplusi = map(i, 0, grid.size(), 0, palette.size() - 1);

    // get color
    RColor col = palette.find(tplusi);

    // draw stripe with color
    fill(col.getColor());
    rect(x - 0.4, y, w + 0.8, h);
  }


  // draw info
  String txt = "Palette sorted by: " + paletteSort[paletteSortIndex] + ", colors mixed in: " + palette.getColorSpace();

  fill(255);
  rect(0, height * 0.5, textWidth(txt) + 20, 20);

  fill(0);
  text(txt, 10, height * 0.5 + 16);
}

void mousePressed() {

  if (mouseButton == LEFT) {
    colorSpaceIndex++;
    if (colorSpaceIndex == spaces.length) {
      colorSpaceIndex = 0;
    }
    palette.setColorSpace(spaces[colorSpaceIndex]);
  } else {
    paletteSortIndex++;
    if (paletteSortIndex == paletteSort.length) {
      paletteSortIndex = 0;
    }
    palette.sort(paletteSort[paletteSortIndex]);
  }
}

void keyPressed() {

  if (key == 's') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  } else if (key == 'd') {
    colorSpaceIndex++;
    if (colorSpaceIndex == spaces.length) {
      colorSpaceIndex = 0;
    }
    palette.setColorSpace(spaces[colorSpaceIndex]);
  } else if (key == 'w') {
    paletteSortIndex++;
    if (paletteSortIndex == paletteSort.length) {
      paletteSortIndex = 0;
    }
    palette.sort(paletteSort[paletteSortIndex]);
  }
}