import net.returnvoid.tools.*;
import net.returnvoid.io.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;

/**
 * This sketch shows how to get a color of a
 * color palette which is similar to an input color. The colors of a 
 * palette are used to color an image (an orginal color is picked 
 * from an image and the most similar color in the color palette
 * is calculated. That found color is used to color the image).
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

// loader for color palettes / images:
// new palettes / images will be loaded when
// pressing the arrow buttons or tab or by clicking
ColorPaletteLoader paletteLoader;
ImageLoader imageLoader;

void setup() {
  size(1280, 720);

  paletteLoader = new ColorPaletteLoader(this);
  imageLoader = new ImageLoader(this);

  frameRate(30);
  textFont(createFont("Arial", 14));
}

void draw() {
  background(0);

  // get color palette / image from loaders
  PImage img = imageLoader.getCurrent().copy();
  img.resize(0, height);
  ColorPalette palette = paletteLoader.getCurrent();

  // create a pixelated version of the image 
  // with colors from the color palette
  TableGrid grid = GridMaker.createTable(0, 0, img.width, img.height, 30000);
  img.loadPixels();
  for (int i = 0; i < grid.size(); i++) {
    float x = grid.getX(i);
    float y = grid.getY(i);
    float w = grid.cellWidth();
    float h = grid.cellHeight();

    // get color from image
    int index = (int) y * img.width + (int) x;
    color imageColor = img.pixels[index];

    // convert image color to color object
    RColor rImageColor = RGBColor.toRGB(imageColor);

    // get most similar color from palettes (to the given image color)
    RColor c = palette.get(rImageColor);

    // draw pixels
    fill(c.getColor());
    stroke(c.getColor());
    rect(x - w * 0.5 + width - img.width, y - h * 0.5, w, h);
  }

  // info panel

  float imageW = 250;

  String imageFileName = imageLoader.getFileName(true);
  String colorFileName =  paletteLoader.getFileName(true);

  if (colorFileName.length() >= 24) {
    colorFileName = colorFileName.substring(0, 24) + "...";
  }

  String infoText = "Image: " + imageFileName + "\n";
  infoText += "Colors: " + colorFileName + "\n";
  infoText += "Press mouse for new colors + image";
  noStroke();
  fill(10);
  rect(0, 0, imageW, 66);
  fill(255);
  text(infoText, 10, 16);


  float imageH = img.height * imageW / img.width;
  image(img, 0, 66, imageW, imageH);
  palette.draw(this, 0, 65 + imageH, imageW, imageH);
}

void mousePressed() {
  paletteLoader.nextRandom();
  imageLoader.nextRandom();
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}