import net.returnvoid.tools.*;
import net.returnvoid.io.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;

/**
 * This sketch shows the effect of different color spaces / 
 * color difference measures on dectecting similarities. Similarities
 * are used to determine the most similar color of a color palette 
 * to an input color. The effect of the measures are displayed by 
 * coloring an images with colors from a palette using two
 * different measures (an orginal color is picked from an image 
 * and the most similar color in the color palette
 * is calculated. That found color is used to color the image).
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */


// loader for color palettes / images:
// new palettes / images will be loaded when
// pressing the arrow buttons or tab or by clicking
ColorPaletteLoader paletteLoader;
ImageLoader imageLoader;

void setup() {
  size(1280, 720);

  paletteLoader = new ColorPaletteLoader(this);
  imageLoader = new ImageLoader(this);

  frameRate(30);
  textFont(createFont("Arial", 14));
}

void draw() {
  background(0);

  // get color palette / image from loaders
  PImage img = imageLoader.getCurrent().copy();
  img.resize(0, height / 2);
  ColorPalette originalPalette = paletteLoader.getCurrent();

  // copy the color palettes and create two new instances
  // loaded color palettes contain always RGB colors
  ColorPalette rgbPalette = originalPalette.copy();
  ColorPalette lchPalette = originalPalette.copy();

  // set the color diffence measure for the two instances
  rgbPalette.setColorDifferenceMeasure(ColorDifferenceMeasure.RGBMeanRed);      // set the similarity calculation by a measure
  lchPalette.setColorSpace(ColorSpace.LCHColor);                                // set the similarity calculation by a color space (measure will be the Euclidean distance of that space)


  // display calculation information
  fill(255);
  textAlign(RIGHT);
  String rgbText = "Difference measure: " + rgbPalette.getColorDifferenceMeasure() 
    + "\nColor space: " + rgbPalette.getColorSpace();
  String lchText = "Difference measure: "  + lchPalette.getColorDifferenceMeasure() 
    + "\nColor space: " + lchPalette.getColorSpace();   

  text(rgbText, width - img.width - 10, 22);
  text(lchText, width - img.width - 10, height * 0.5 + 22);


  // create a pixelated version of the image 
  // with colors from the color palette
  TableGrid grid = GridMaker.createTable(0, 0, img.width, img.height, 10000);
  img.loadPixels();
  for (int i = 0; i < grid.size(); i++) {
    float x = grid.getX(i);
    float y = grid.getY(i);
    float w = grid.cellWidth();
    float h = grid.cellHeight();
    float xx = width - img.width + x - w * 0.5;
    float yy = y - h * 0.5;

    // get color from image
    int index = (int) y * img.width + (int) x;
    color imageColor = img.pixels[index];

    // convert image color to color object
    RGBColor rImageColor = RGBColor.toRGB(imageColor);

    // get most similar color from palettes (to the given image color)
    RColor rgbColor = rgbPalette.get(rImageColor);
    RColor lchColor = lchPalette.get(rImageColor);

    // draw pixels
    fill(rgbColor.getColor());
    stroke(rgbColor.getColor());
    rect(xx, yy, w, h);

    fill(lchColor.getColor());
    stroke(lchColor.getColor());
    rect(xx, yy + height * 0.5, w, h);
  }

  // info panel on the left 

  float imageW = 300;
  float imageH = img.height * imageW / img.width;

  String imageFileName = imageLoader.getFileName(true);
  String colorFileName =  paletteLoader.getFileName(true);

  if (colorFileName.length() >= 24) {
    colorFileName = colorFileName.substring(0, 24) + "...";
  }

  String infoText = "Image: " + imageFileName + "\n";
  infoText += "Colors: " + colorFileName + "\n";
  infoText += "Press mouse for new colors + image";
  textAlign(LEFT);
  noStroke();
  fill(10);
  rect(0, 0, imageW, 66);
  fill(255);
  text(infoText, 10, 16);



  image(img, 0, 66, imageW, imageH);
  originalPalette.draw(this, 0, 65 + imageH, imageW, imageH);
}

void mousePressed() {
  paletteLoader.nextRandom();
  imageLoader.nextRandom();
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}