import net.returnvoid.tools.*;
import net.returnvoid.io.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;

/**
 * This sketch shows how to reduce the number of
 * colors of a palette without deleting colors but
 * by merging similar colors.
 * Additional the effect of different color spaces /
 * color difference measures on the merging process
 * is displayed by showing the color reduction 
 * result with two color models.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */


// loader for color palettes:
// new palettes will be loaded when
// pressing the arrow buttons or tab or by clicking
ColorPaletteLoader paletteLoader;

void setup() {
  size(1280, 720);

  paletteLoader = new ColorPaletteLoader(this);

  frameRate(30);

  textFont(createFont("Arial", 16));
}

void draw() {
  background(0);

  // get color palette from loader
  ColorPalette originalPalette = paletteLoader.getCurrent();

  // copy the color palettes and create two new instances
  // loaded color palettes contain always RGB colors
  ColorPalette reducedPaletteRGB = originalPalette.copy();
  ColorPalette reducedPaletteLCH = originalPalette.copy();

  // reduce the number of colors with RGB colors (and sort the result by luminance)
  reducedPaletteRGB.reduceSize(5);
  reducedPaletteRGB.sort();

  // reduce the number of colors with LCH colors (and sort the result by luminance)
  reducedPaletteLCH.setColorSpace(ColorSpace.LCHColor);
  reducedPaletteLCH.reduceSize(5);
  reducedPaletteLCH.sort();

  // draw original palett and the reduced versions
  originalPalette.draw(this, 0, 0, width, height * 0.33333);
  reducedPaletteRGB.draw(this, 0, height * 0.33333, width, height * 0.33333);
  reducedPaletteLCH.draw(this, 0, 2 * height * 0.33333, width, height * 0.33333);

  // draw info text
  String textOriginal = "Input color palette - " + originalPalette.size() + " colors";
  String textRGB = "Reduced color palette using RGBColor - " + reducedPaletteRGB.size() + " colors";
  String textLCH = "Reduced color palette using LCHColor - " + reducedPaletteLCH.size() + " colors";
  noStroke();
  fill(255);
  rect(0, 0, textWidth(textOriginal) + 20, 22);
  rect(0, height * 0.33333, textWidth(textRGB) + 20, 22);
  rect(0, 2 * height * 0.33333, textWidth(textLCH) + 20, 22);
  fill(0);
  text(textOriginal, 10, 16);
  text(textRGB, 10, height * 0.33333 + 16);
  text(textLCH, 10, 2 * height * 0.33333 + 16);
}

void mousePressed() {
  // get a new random color palette
  paletteLoader.nextRandom();
}

void keyPressed() {
  if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  }
}