import net.returnvoid.color.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows the basic usage of color palettes including:
 * - add colors to a palette
 * - set importance / weighting for each color in the palette
 * - draw the palette
 * - iterate over all colors of the palette
 * - get colors from the palette randomly
 * - get colors from the palette randomly based on their importance
 * - save the palette to color.json
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

ColorPalette palette;

void setup() {
  size(1280, 720);

  // create an empty color palette
  palette = new ColorPalette();

  // add colors to the palette with their importance
  palette.add(#000000, 8);
  palette.add(#FFFFFF, 1);
  palette.add(#DBC299, 1);
  palette.add(#B4217E, 2);
  palette.add(#89134E, 2);

  // set a name and tags to the palette
  // names and tags will be saved to the color.json when doing the export
  palette.setName("magentas");
  palette.addTags("magenta", "black", "white", "high contrast");

  noLoop();

  textFont(createFont("Arial", 16));
}

void draw() {

  background(255);
  textAlign(LEFT, TOP);

  // (1) first row: use draw() method of the palette -------------------------------------------

  // you can draw the entire palette with this single method call
  // colors will be displayed with vertical striped in the given
  // rectangular area
  palette.draw(this, 0, 0, width, height);

  // draw info text 
  String drawText = "(1) palette.draw()";
  noStroke();
  fill(255);
  rect(0, 0, textWidth(drawText) + 20, 22);
  fill(0);
  text(drawText, 10, 0);

  // (2) second row: iterate over all colors of the palette -------------------------------------------

  // palette.size() returns the number of colors in the palette
  for (int i = 0; i < palette.size(); i++) {
    float w = (float) width / palette.size();
    float h = height * 0.25;
    float x = i * w;
    float y = h;

    // get the color object from the palette
    RColor cObj = palette.get(i);

    // and convert it to Processing's color
    color c = cObj.getColor();

    // draw color with rectangle
    fill(c);
    stroke(c);
    rect(x, y, w, h);

    // get and show weighing of the color
    // all weightings of a color palette sums up to 1
    float weighting = palette.getWeighting(cObj);
    String weightingText = nf(weighting * 100, 3, 2) + "%";
    noStroke();
    fill(255, 120);
    rect(x, y + h - 22, w, 22);
    fill(0);
    text(weightingText, x + 10, y + h - 20);
  }

  // info text
  String iterateText = "(2) iterate over all colors of a palette";
  noStroke();
  fill(255);
  rect(0, height * 0.25, textWidth(iterateText) + 20, 22);
  fill(0);
  text(iterateText, 10, height * 0.25);

  // (3) third row: get colors from palette randomly -------------------------------------------


  int randomColors = 200;
  for (int i = 0; i < randomColors; i++) {
    float w = (float) width / randomColors;
    float h = height * 0.25;
    float x = i * w;
    float y = h * 2;

    // get the color object from the palette randomly
    // all colors will be returned equally often (approx.)
    RColor cObj = palette.getRandom();

    //  convert it to Processing's color
    color c = cObj.getColor();

    fill(c);
    stroke(c);
    rect(x, y, w, h);
  }

  // info text
  String randomText = "(3) get colors from palette randomly";
  noStroke();
  fill(255);
  rect(0, height * 0.5, textWidth(randomText) + 20, 22);
  fill(0);
  text(randomText, 10, height * 0.5);

  // (4) fourth row: get colors from palette randomly by weighting -------------------------------------------

  for (int i = 0; i < randomColors; i++) {
    float w = (float) width / randomColors;
    float h = height * 0.25;
    float x = i * w;
    float y = h * 3;

    // get the color object from the palette randomly
    // more important colors will be returned more often the less
    // important colors
    RColor cObj = palette.getRandomWeighted();

    //  convert it to Processing's color
    color c = cObj.getColor();

    fill(c);
    stroke(c);
    rect(x, y, w, h);
  }

  // info text
  String randomWeightedText = "(4) get colors from palette randomly by weighting";
  noStroke();
  fill(255);
  rect(0, height * 0.75, textWidth(randomWeightedText) + 20, 22);
  fill(0);
  text(randomWeightedText, 10, height * 0.75);
}

void mousePressed() {
  redraw();
}

void keyPressed() {
  if (key == 's') {
    // save the color palette in sub-folder 'palettes' with name 'themagentas'
    palette.save(this, "palettes", "themagentas");
    println("palette saved");
  } else if (key == 'p') {
    saveFrame("export/" + StringTools.timestamp() + ".png");
  } else {

    redraw();
  }
}