import net.returnvoid.analytics.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;
import net.returnvoid.io.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows how to use the ColorClusteringBuilder to do clusterings and how to get the optimal clustering of a given img / of given colors.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

ImageLoader loader;

void setup() {
  size(1280, 720);

  loader = new ImageLoader(this);
  noLoop();

  textFont(createFont("Arial", 16));
}

void draw() {

  background(10);

  // get an image
  PImage img = loader.nextRandom();

  // set a color difference measure. the measure effects the quality of the clustering (detecting similarities of colors)
  ColorDifferenceMeasure measure1 = ColorDifferenceMeasure.LCHEuclidean;
  ColorDifferenceMeasure measure2 = ColorDifferenceMeasure.RGBEuclidean;

  // draw image
  float imgHeight = 200;
  float imgWidth = img.width * imgHeight / img.height;

  image(img, width * 0.5 - imgWidth * 0.5, 0, imgWidth, imgHeight);


  // ColorClusteringBuilder can either be used by setting an image or by setting colors
  // the clustering will start after setImage() or setColors() is called
  // the optimal clustering will be found in the range of the set minimal k and maximal k 
  // the default value for minimal k = 2, for maximal k = 30

  // create builder instances 
  ColorClusteringBuilder builder1 = new ColorClusteringBuilder(this, measure1);
  ColorClusteringBuilder builder2 = new ColorClusteringBuilder(this, measure2);

  // set maximal k: mybuilder.setMaxK(int)

  // provide one builder with an image (training colors will be detected by the builder)
  // provide the other builder with a set of colors
  builder1.setImage(img);
  
  // get the training colors which are used for clustering of first builder. These colors
  // will also be used for the clustering of the second builder
  RColor[] colors = builder1.getColors();
  builder2.setColors(colors);

  // get clusterings with optimal k for both builders 
  ColorClustering optimalofbuilder1 = builder1.getClustering();
  ColorClustering optimalofbuilder2 = builder2.getClustering();

  // get clusterings with k=2 for both builders
  ColorClustering k2ofbuilder1 = builder1.getClustering(2);
  ColorClustering k2ofbuilder2 = builder2.getClustering(2);

  // convert all clusterings to color palettes
  ColorPalette palleteOptimal1 = optimalofbuilder1.toColorPalette();
  ColorPalette palleteOptimal2 = optimalofbuilder2.toColorPalette();

  ColorPalette palletek21 = k2ofbuilder1.toColorPalette();
  ColorPalette palletek22 = k2ofbuilder2.toColorPalette();
  
  // draw all palettes (equals to found clusterings)
  float paletteHeight = 0.5 * (height - imgHeight);
  float paletteWidth = width * 0.5;

  palleteOptimal1.draw(this, 0, imgHeight, paletteWidth, paletteHeight);
  palletek21.draw(this, 0, imgHeight + paletteHeight, paletteWidth, paletteHeight);

  palleteOptimal2.draw(this, paletteWidth, imgHeight, paletteWidth, paletteHeight);
  palletek22.draw(this, paletteWidth, imgHeight + paletteHeight, paletteWidth, paletteHeight);

  // draw builder information
  textAlign(LEFT, TOP);
  fill(255);
  String txt1 = "Clusterings build with:\n"; 
  txt1 += builder1.getColorDifferenceMeasure() + "\n";
  txt1 += "detected optimal k: " + builder1.getOptimalK();
  text(txt1, 10, 10);

  textAlign(RIGHT, TOP);
  String txt2 = "Clusterings build with:\n"; 
  txt2 += builder2.getColorDifferenceMeasure() + "\n";
  txt2 += "detected optimal k: " + builder2.getOptimalK();
  text(txt2, width - 10, 10);
}

void mousePressed() {
  redraw();
}


void keyPressed() {
  saveFrame("export/" + StringTools.timestamp() + ".png");
}