import net.returnvoid.analytics.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;
import net.returnvoid.io.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows how to create multiple clusterings of colors with a given range of k (number of colors in each clustering).
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

ImageLoader loader;

void setup() {
  size(1280, 720);

  loader = new ImageLoader(this);
  noLoop();

  textFont(createFont("Arial", 14));
}

void draw() {

  // get an image
  PImage img = loader.nextRandom();

  // set a color difference measure. the measure effects the quality of the clustering (detecting similarities of colors)
  ColorDifferenceMeasure measure = ColorDifferenceMeasure.LCHEuclidean;

  // get 420 sampled colors from the image
  RColor[] imgColors = ClusteringHelper.getTrainingColors(img, 420, measure);

  // draw the sampled colors (left row)
  TableGrid grid1 = GridMaker.createTable(0, 0, width * 0.25, height, 1, imgColors.length);
  for (int i = 0; i < grid1.size(); i++) {
    float w = grid1.getWidth(i);
    float h = grid1.getHeight(i);
    float x = grid1.getX(i);
    float y = grid1.getY(i);

    color c = imgColors[i].getColor();

    fill(c);
    stroke(c);
    rect(x - w * 0.5, y - h * 0.5, w, h);
  } 

  // set range of k (mink => clustering with smallest number of colors, maxk => clustering with highest number of colors, k => number of colors for clustering)
  int mink = 2;
  int maxk = 10;

  // learn clusterings with given range of k (will return maxk - mink + 1 clusterings)
  ColorClustering[] clusterings = ClusteringHelper.kmeansClusterings(imgColors, mink, maxk, measure);

  // draw all clusterings (right row)
  TableGrid clusteringGrid = GridMaker.createTable(width * 0.25, 0, width * 0.75, height, 1, clusterings.length);
  for (int i = 0; i < clusteringGrid.size(); i++) {
    float w = clusteringGrid.getWidth(i);
    float h = clusteringGrid.getHeight(i);
    float x = clusteringGrid.getX(i) - w * 0.5;
    float y = clusteringGrid.getY(i) - h * 0.5;
    
    // get current clustering
    ColorClustering clustering = clusterings[i];
    
    // get k of current clustering (number of colors / clusters in that clustering)
    int k = clustering.getK();

    // convert the clustering to a color palette and draw it
    ColorPalette palette = clustering.toColorPalette();
    palette.draw(this, x, y, w, h);

    // draw loss
    float loss = clustering.getLoss();

    fill(0, 60);
    rect(x, y, w, 20);

    fill(255);
    textAlign(CENTER, TOP);
    text("Loss / Error of clustering (k=" + k + "): " + nfs(loss, 3, 3), x + w * 0.5, y + 2);
  }
}

void mousePressed() {
  redraw();
}


void keyPressed() {
  saveFrame("export/" + StringTools.timestamp() + ".png");
}