import net.returnvoid.analytics.*;
import net.returnvoid.color.*;
import net.returnvoid.graphics.grid.*;
import net.returnvoid.io.*;
import net.returnvoid.tools.*;

/**
 * This sketch shows how to create a clustering of colors with a given number of clusters.
 *
 * This code is copyright (c) Diana Lange 2017
 *
 * The library is published under the Creative Commons license NonCommercial 4.0.
 * Please check https://creativecommons.org/licenses/by-nc/4.0/ for more information.
 * 
 * This program is distributed in the hope that it will be useful, but without any warranty.
 */

ImageLoader loader;

void setup() {
  size(1280, 720);

  loader = new ImageLoader(this);
  noLoop();

  textFont(createFont("Arial", 16));
}

void draw() {

  background(10);

  // get an image
  PImage img = loader.nextRandom();

  // set a color difference measure. the measure effects the quality of the clustering (detecting similarities of colors)
  ColorDifferenceMeasure measure = ColorDifferenceMeasure.LCHEuclidean;

  // get 420 sampled colors from the image
  RColor[] colors = ClusteringHelper.getTrainingColors(img, 420, measure);

  // draw the sampled colors (left row)
  TableGrid grid1 = GridMaker.createTable(0, 0, width, height * 0.5, colors.length, 1);
  for (int i = 0; i < grid1.size(); i++) {
    float w = grid1.getWidth(i);
    float h = grid1.getHeight(i);
    float x = grid1.getX(i);
    float y = grid1.getY(i);

    color c = colors[i].getColor();

    fill(c);
    stroke(c);
    rect(x - w * 0.5, y - h * 0.5, w, h);
  }

  // k equals to the number of colors which will be extrackted
  int k = 7;

  // learn clustering with k number of cluster
  ColorClustering clustering = ClusteringHelper.kmeans(colors, k, measure);

  // convert the clustering to a color palette and draw it
  ColorPalette palette = clustering.toColorPalette();
  palette.draw(this, 0, height * 0.5, width, height * 0.5);

  // draw loss
  float loss = clustering.getLoss();

  fill(0, 60);
  rect(0, height - 40, width, 40);

  fill(255);
  textAlign(CENTER, BOTTOM);
  text("Loss / Error of clustering (k=" + k + "): " + nfs(loss, 3, 3), width * 0.5, height - 10);
}

void mousePressed() {
  redraw();
}


void keyPressed() {
  saveFrame("export/" + StringTools.timestamp() + ".png");
}